<?php
// chat_module/install.php
// Web installer for the chat module.
// 1) Upload the whole chat_module folder to your server (e.g. /public_html/chat_module)
// 2) Open https://YOUR-DOMAIN/chat_module/install.php in your browser
// 3) Fill in your MySQL details (same DB where your users table lives)

session_start();

$configFile = __DIR__ . '/config.php';

// If config already exists, try to load it so we can reuse DB settings
if (file_exists($configFile)) {
    require_once $configFile;
}

function h($v) {
    return htmlspecialchars((string)$v, ENT_QUOTES, 'UTF-8');
}

$errors = [];
$success = false;

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $dbHost = trim($_POST['db_host'] ?? 'localhost');
    $dbName = trim($_POST['db_name'] ?? '');
    $dbUser = trim($_POST['db_user'] ?? '');
    $dbPass = (string)($_POST['db_pass'] ?? '');

    if ($dbHost === '' || $dbName === '' || $dbUser === '') {
        $errors[] = 'Please fill in all required fields (host, database, user).';
    } else {
        $dsn = 'mysql:host=' . $dbHost . ';dbname=' . $dbName . ';charset=utf8mb4';
        try {
            $pdo = new PDO($dsn, $dbUser, $dbPass, [
                PDO::ATTR_ERRMODE            => PDO::ERRMODE_EXCEPTION,
                PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
            ]);
        } catch (Throwable $e) {
            $errors[] = 'Database connection failed: ' . $e->getMessage();
            $pdo = null;
        }

        if ($pdo) {
            // Create tables
            $sql = file_get_contents(__DIR__ . '/sql/create_chat_tables.sql');
            if ($sql === false) {
                $errors[] = 'Could not read SQL file.';
            } else {
                $chunks = array_filter(array_map('trim', explode(';', $sql)));
                foreach ($chunks as $chunk) {
                    if ($chunk === '') continue;
                    $stmtSql = $chunk . ';';
                    try {
                        $pdo->exec($stmtSql);
                    } catch (Throwable $e) {
                        $errors[] = 'SQL error: ' . $e->getMessage();
                    }
                }
            }

            if (!$errors) {
                // Write config.php
                $cfg = "<?php\n"
                     . "define('CHAT_DB_HOST', " . var_export($dbHost, true) . ");\n"
                     . "define('CHAT_DB_NAME', " . var_export($dbName, true) . ");\n"
                     . "define('CHAT_DB_USER', " . var_export($dbUser, true) . ");\n"
                     . "define('CHAT_DB_PASS', " . var_export($dbPass, true) . ");\n\n"
                     . "function chat_db(): PDO {\n"
                     . "    static $pdo = null;\n"
                     . "    if ($pdo instanceof PDO) {\n"
                     . "        return $pdo;\n"
                     . "    }\n"
                     . "    $dsn = 'mysql:host=' . CHAT_DB_HOST . ';dbname=' . CHAT_DB_NAME . ';charset=utf8mb4';\n"
                     . "    $pdo = new PDO($dsn, CHAT_DB_USER, CHAT_DB_PASS, [\n"
                     . "        PDO::ATTR_ERRMODE            => PDO::ERRMODE_EXCEPTION,\n"
                     . "        PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,\n"
                     . "    ]);\n"
                     . "    return $pdo;\n"
                     . "}\n";

                if (file_put_contents($configFile, $cfg) === false) {
                    $errors[] = 'Could not write config.php (check file permissions).';
                } else {
                    $success = true;
                }
            }
        }
    }
}

$dbHostDefault = defined('CHAT_DB_HOST') ? CHAT_DB_HOST : 'localhost';
$dbNameDefault = defined('CHAT_DB_NAME') ? CHAT_DB_NAME : '';
$dbUserDefault = defined('CHAT_DB_USER') ? CHAT_DB_USER : '';
?>
<!doctype html>
<html lang="en">
<head>
  <meta charset="utf-8">
  <title>Chat Module Installer</title>
  <meta name="viewport" content="width=device-width,initial-scale=1">
  <style>
    body { font-family: system-ui, -apple-system, BlinkMacSystemFont, "Segoe UI", sans-serif; background:#f5f6fa; margin:0; padding:20px; }
    .card { max-width:700px; margin:20px auto; background:#fff; border-radius:8px; padding:20px; box-shadow:0 2px 10px rgba(0,0,0,0.06); }
    h1 { font-size:22px; margin-top:0; }
    .field { margin-bottom:12px; }
    label { display:block; font-size:13px; margin-bottom:4px; color:#374151; }
    input[type=text], input[type=password] { width:100%; padding:8px; border-radius:6px; border:1px solid #cbd5e1; font-size:14px; }
    .btn { display:inline-block; padding:8px 16px; border-radius:6px; border:none; background:#2563eb; color:#fff; cursor:pointer; font-size:14px; }
    .btn:disabled { opacity:0.6; cursor:not-allowed; }
    .errors { margin:10px 0; padding:10px; background:#fee2e2; color:#b91c1c; border-radius:6px; font-size:13px; }
    .success { margin:10px 0; padding:10px; background:#dcfce7; color:#166534; border-radius:6px; font-size:13px; }
    code { background:#f3f4f6; padding:2px 4px; border-radius:4px; }
    ol { padding-left:20px; }
  </style>
</head>
<body>
<div class="card">
  <h1>Chat Module Installer</h1>

  <?php if ($success): ?>
    <div class="success">
      <strong>Success!</strong> Tables were created and <code>config.php</code> was written.
    </div>
    <h2>Next steps</h2>
    <ol>
      <li>For security, you may now delete <code>install.php</code> from the <code>chat_module</code> folder.</li>
      <li>Make sure the rest of the module files are in <code>/chat_module</code>:
        <ul>
          <li><code>assets/css/chat.css</code></li>
          <li><code>assets/js/chat.js</code></li>
          <li><code>api/chat/*.php</code></li>
          <li><code>public/chat_list_demo.php</code></li>
        </ul>
      </li>
      <li>Open <code>/chat_module/public/chat_list_demo.php</code> in your browser while logged in to your site.</li>
    </ol>
  <?php else: ?>
    <?php if ($errors): ?>
      <div class="errors">
        <strong>Installation errors:</strong>
        <ul>
          <?php foreach ($errors as $e): ?>
            <li><?php echo h($e); ?></li>
          <?php endforeach; ?>
        </ul>
      </div>
    <?php endif; ?>

    <p>Enter your MySQL details. Use the same database where your <code>users</code> table is stored.</p>
    <form method="post">
      <div class="field">
        <label for="db_host">DB Host</label>
        <input type="text" id="db_host" name="db_host" value="<?php echo h($dbHostDefault); ?>" required>
      </div>
      <div class="field">
        <label for="db_name">DB Name</label>
        <input type="text" id="db_name" name="db_name" value="<?php echo h($dbNameDefault); ?>" required>
      </div>
      <div class="field">
        <label for="db_user">DB User</label>
        <input type="text" id="db_user" name="db_user" value="<?php echo h($dbUserDefault); ?>" required>
      </div>
      <div class="field">
        <label for="db_pass">DB Password</label>
        <input type="password" id="db_pass" name="db_pass" value="">
      </div>
      <button type="submit" class="btn">Install</button>
    </form>
  <?php endif; ?>
</div>
</body>
</html>
